{
   p2dvideo.pas

   Copyright 2013-2015 Markus Mangold <info@retrogamecoding.org>


  This software is provided 'as-is', without any express or implied
  warranty. In no event will the authors be held liable for any damages
  arising from the use of this software.

  Permission is granted to anyone to use this software for any purpose,
  including commercial applications, and to alter it and redistribute it
  freely, subject to the following restrictions:

   1. The origin of this software must not be misrepresented; you must not
   claim that you wrote the original software. If you use this software
   in a product, an acknowledgment in the product documentation would be
   appreciated but is not required.

   2. Altered source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

   3. This notice may not be removed or altered from any source
   distribution.

}

{$DEFINE useaudio}

unit P2DVideo;

interface
uses
{$IFDEF windows}
  Windows,
{$ENDIF}
  SysUtils,
  SDL2,
{$IFDEF useaudio}
  SDL2_mixer,
{$ENDIF}
  SDL2_image;

{$I fontdata.inc}

type
  p2dsprite = PSDL_Texture;
type
  p2dwindow = integer;
type
  p2dimage = PSDL_Surface;
type
  p2dfont = integer;

var
  activewindow, fontsize, fonttype: integer;
  colorkey: boolean;
  fullscreenmode, virtualsize: boolean;
  joystick: PSDL_Joystick;

function openwindow(caption: string; x, y, w, h: integer): integer;
function windowheight(): integer;
function windowwidth(): integer;
procedure closewindow(window: integer);
procedure setactivewindow(windowID: integer);
procedure setvirtualsize(w, h: integer);
procedure togglefullscreen;
function fullscreen: boolean;
procedure setscalequality(mode: string);
procedure setcaption(caption: string);
procedure seticon(icon: PSDL_Surface);
procedure resizewindow(x, y: integer);

function createsprite(image: PSDL_Surface): PSDL_Texture;
function loadsprite(filename: string): PSDL_Texture;
procedure drawsprite(sprite: PSDL_Texture; x, y: integer; width, height, angle: real; vflip, hflip: boolean);
procedure drawspritepart(sprite: PSDL_Texture; x, y, startx, starty, xoffset, yoffset: integer; width, height, angle:
  real; vflip, hflip: boolean);
function spritewidth(sprite: PSDL_Texture): integer;
function spriteheight(sprite: PSDL_Texture): integer;

procedure clearwindow;
procedure color(red, green, blue, alpha: integer);
procedure backcolor(red, green, blue, alpha: integer);
procedure redraw;
procedure wait(ms: integer);
function loadimage(filename: string): PSDL_Surface;
procedure setcolorkey(red, green, blue: integer);
procedure nocolorkey;
procedure freeimage(srcimage: PSDL_Surface);
procedure freesprite(srcsprite: PSDL_Texture);
procedure closeapplication;
procedure spritecolor(tex: PSDL_Texture; red, green, blue, alpha: integer);

procedure SetFrameTimer(frames: integer);
function TimeLeft: UInt32;
procedure sync();

procedure Messagebox(typus: integer; title, textmessage: string);
function timerticks: integer;

procedure fillrectangle(x, y, w, h: integer);
procedure rectangle(x, y, w, h: integer);
procedure dot(x, y: integer);
procedure line(x1, y1, x2, y2: integer);
procedure Circle(xcenter, ycenter, radius: integer);
procedure fillcircle(cx, cy, r: integer);
procedure Ellipse(CX, CY, XRadius, YRadius: longint);
procedure fillellipse(xc, yc, width, height: integer);
procedure triangle(x1, y1, x2, y2, x3, y3: integer);
procedure filltriangle(xa, ya, xb, yb, xc, yc: integer);

procedure drawchar(x, y: integer; ch: char);
procedure textsize(size: integer);
procedure texttype(typus: integer);
procedure drawtext(str: string; x, y: integer);
function textwidth(TextString: string): integer;
function textheight(): integer;
procedure textinputbackcolor(red, green, blue, alpha: integer);
procedure textinputcolor(red, green, blue, alpha: integer);
function input(x, y, maxlen: integer): string;

function createbmpfont(sprite: p2dsprite; width, height: integer; fontface: ansistring): integer;
procedure bmptext(textstr: ansistring; xspace, yspace: integer; fontnumber: integer);
procedure bmpfontsize(fontnumber: integer; xf, yf: integer);
procedure bmpfontangle(fontnumber: integer; angle: real);
procedure bmpinputbackcolor(fontnumber, red, green, blue, alpha: integer);
procedure bmpinputcolor(fontnumber, red, green, blue, alpha: integer);
function bmpinput(fontnumber, x, y, maxlen: integer): string;
function bmpfontheight(fontnumber: integer): integer;
function bmpfontwidth(fontnumber: integer): integer;

implementation

type
  bitmapfont = record
    image: array of p2dsprite;
    width, height, lines, rows: array of integer;
    xfactor, yfactor: array of integer;
    binpred, binpgreen, binpblue, binpalpha: array of integer;
    inpred, inpgreen, inpblue, inpalpha: array of integer;
    angle: array of real;
    index: integer;
    face: array of ansistring;
  end;

var
  bkr, bkg, bkb, bka, ckr, ckg, ckb, tbred, tbgreen, tbblue, tbalpha, tcred, tcgreen, tcblue, tcalpha: integer;
  TICK_INTERVAL: integer = 1000 div 100;
  next_time: cardinal = 0;
  bmpfont: bitmapfont;

  //this opens a window and a connected renderer

function openwindow(caption: string; x, y, w, h: integer): integer;
var
  window: PSDL_Window;
  renderer: PSDL_Renderer;
begin
  if x < 0 then
  begin
    x := SDL_WINDOWPOS_CENTERED;
  end;
  if y < 0 then
  begin
    y := SDL_WINDOWPOS_CENTERED;
  end;
  window := SDL_CreateWindow(PChar(caption), x, y, w, h, SDL_WINDOW_SHOWN);
  if (window = nil) then
  begin
    Messagebox(1, 'Error', 'SDL_CreateWindow Error: ' + SDL_GetError);
    SDL_Quit;
    Halt;
  end;
  renderer := SDL_CreateRenderer(window, -1, 0);
  if (renderer = nil) then
  begin
    Messagebox(1, 'Error', 'SDL_CreateRenderer Error: ' + SDL_GetError);
    SDL_Quit;
    Halt;
  end;
  result := SDL_GetWIndowID(window);
end;

//setes the virtualsize for the game

procedure setvirtualsize(w, h: integer);
begin
  SDL_RenderSetLogicalSize(SDL_GetRenderer(SDL_GetWindowFromID(activewindow)), w, h);
  virtualsize := true;
end;

//closes the given window

procedure closewindow(window: integer);
begin
  SDL_DestroyRenderer(SDL_GetRenderer(SDL_GetWindowFromID(window)));
  SDL_DestroyWindow(SDL_GetWindowFromID(window));
  fullscreenmode := false;
end;

//resizes the given window

procedure resizewindow(x, y: integer);
begin
  if not fullscreen then
  begin
    SDL_SetWindowSize(SDL_GetWindowFromID(activewindow), x, y);
  end;
end;

//set render scale quality - look into SDL documentation

procedure setscalequality(mode: string);
begin
  SDL_SetHint(SDL_HINT_RENDER_SCALE_QUALITY, PChar(mode));
end;

//sets the active window for all following functions

procedure setactivewindow(windowID: integer);
begin
  activewindow := windowID;
end;

//sets the global colourkey for images

procedure setcolorkey(red, green, blue: integer);
begin
  ckr := red;
  ckg := green;
  ckb := blue;
  colorkey := true;
end;

//turns colourkey off

procedure nocolorkey;
begin
  colorkey := false;
end;

//creates a sprite (texture) out of an image

function createsprite(image: PSDL_Surface): PSDL_Texture;
var
  textu: PSDL_Texture;
begin
  textu := SDL_CreateTextureFromSurface(SDL_GetRenderer(SDL_GetWindowFromID(activewindow)), image);
  if (textu = nil) then
  begin
    Messagebox(1, 'Error', 'SDL_CreateTextureFromSurface Error: ' + SDL_GetError);
    SDL_Quit;
    Halt;
  end;
  result := textu;
end;

function loadsprite(filename: string): PSDL_Texture;
var
  sprite: PSDL_Texture;
begin
  if fileexists(filename) then
  begin
    sprite := IMG_LoadTexture(SDL_GetRenderer(SDL_GetWindowFromID(activewindow)), PChar(filename));
  end
  else
  begin
    SDL_ShowSimpleMessageBox(SDL_MESSAGEBOX_ERROR, 'File not found', PCHAR('Can''t find image file: ' + filename), nil);
    SDL_Quit();
    Halt;
  end;
  if (sprite = nil) then
  begin
    Messagebox(1, 'Error', 'SDL_LoadBMP Error: ' + SDL_GetError);
    SDL_Quit;
    Halt;
  end;
  result := sprite;
end;

//draws the sprite onto the active window

procedure drawsprite(sprite: PSDL_Texture; x, y: integer; width, height, angle: real; vflip, hflip: boolean);
var
  dest: PSDL_Rect;
  flip: {TSDL_RendererFlip} byte;
  w, h: integer;
begin
  SDL_QueryTexture(sprite, nil, nil, @w, @h);
  new(dest);
  dest.x := x;
  dest.y := y;
  dest.w := round(w * width);
  dest.h := round(h * height);

  flip := SDL_FLIP_NONE;
  if vflip then
  begin
    flip := flip or SDL_FLIP_VERTICAL;
  end
  else if hflip then
  begin
    flip := flip or SDL_FLIP_HORIZONTAL;
  end;
  SDL_RenderCopyEx(SDL_GetRenderer(SDL_GetWindowFromID(activewindow)), sprite, nil, dest, angle, nil, flip);
  dispose(dest);
end;
//draws a part of the sprite onto the active window

procedure drawspritepart(sprite: PSDL_Texture; x, y, startx, starty, xoffset, yoffset: integer; width, height, angle:
  real; vflip, hflip: boolean);
var
  dest: PSDL_Rect;
  src: PSDL_Rect;
  flip: {TSDL_RendererFlip} byte;
begin
  new(dest);
  dest.x := x;
  dest.y := y;
  dest.w := round(width * (xoffset));
  dest.h := round(height * (yoffset));

  new(src);
  src.x := startx;
  src.y := starty;
  src.w := xoffset;
  src.h := yoffset;

  flip := SDL_FLIP_NONE;

  if vflip then
  begin
    flip := flip or SDL_FLIP_VERTICAL;
  end
  else if hflip then
  begin
    flip := flip or SDL_FLIP_HORIZONTAL;
  end;

  SDL_RenderCopyEx(SDL_GetRenderer(SDL_GetWindowFromID(activewindow)), sprite, src, dest, angle, nil, flip);
  dispose(dest);
  dispose(src);
end;

//draws a filled rectangle

procedure fillrectangle(x, y, w, h: integer);
var
  coord: PSDL_Rect;
begin
  new(coord);
  coord.x := x;
  coord.y := y;
  coord.w := w;
  coord.h := h;
  SDL_RenderFillRect(SDL_GetRenderer(SDL_GetWindowFromID(activewindow)), coord);
  dispose(coord);
end;

//draws a rectangle

procedure rectangle(x, y, w, h: integer);
var
  coord: PSDL_Rect;
begin
  new(coord);
  coord.x := x;
  coord.y := y;
  coord.w := w;
  coord.h := h;
  SDL_RenderDrawRect(SDL_GetRenderer(SDL_GetWindowFromID(activewindow)), coord);
  dispose(coord);
end;

//returns the width of the active window and keeps care of the virtual size

function windowwidth(): integer;
var
  w, h: integer;
begin
  if virtualsize = true then
  begin
    SDL_RenderGetLogicalSize(SDL_GetRenderer(SDL_GetWindowFromID(activewindow)), @w, @h);
  end
  else
  begin
    SDL_GetWindowSize(SDL_GetWindowFromID(activewindow), @w, @h);
  end;
  result := w;
end;

//returns the height of the active window and keeps care of the virtual size

function windowheight(): integer;
var
  w, h: integer;
begin
  if virtualsize = true then
  begin
    SDL_RenderGetLogicalSize(SDL_GetRenderer(SDL_GetWindowFromID(activewindow)), @w, @h);
  end
  else
  begin
    SDL_GetWindowSize(SDL_GetWindowFromID(activewindow), @w, @h);
  end;
  result := h;
end;

//changes caption of active window

procedure setcaption(caption: string);
begin
  SDL_SetWindowTitle(SDL_GetWindowFromID(activewindow), PChar(caption));
end;

//sets an icon for the active window

procedure seticon(icon: PSDL_Surface);
begin
  SDL_SetWindowIcon(SDL_GetWindowFromID(activewindow), icon);
end;

//returns the width of a sprite

function spritewidth(sprite: PSDL_Texture): integer;
var
  w, h: integer;
begin
  SDL_QueryTexture(sprite, nil, nil, @w, @h);
  result := w;
end;

//returns the height of a sprite

function spriteheight(sprite: PSDL_Texture): integer;
var
  w, h: integer;
begin
  SDL_QueryTexture(sprite, nil, nil, @w, @h);
  result := h;
end;

//draws a dot

procedure dot(x, y: integer);
begin
  SDL_RenderDrawPoint(SDL_GetRenderer(SDL_GetWindowFromID(activewindow)), x, y);
end;

//draws a line

procedure line(x1, y1, x2, y2: integer);
begin
  SDL_RenderDrawLine(SDL_GetRenderer(SDL_GetWindowFromID(activewindow)), x1, y1, x2, y2);
end;

//draws a circle

procedure Circle(xcenter, ycenter, radius: integer);
var
  x, y, r2: integer;
begin
  if radius = 0 then
  begin
    radius := 1;
  end;
  r2 := radius * radius;
  dot(xcenter, ycenter + radius);
  dot(xcenter, ycenter - radius);
  dot(xcenter + radius, ycenter);
  dot(xcenter - radius, ycenter);
  x := 1;
  y := round(sqrt(r2 - 1) + 0.5);
  while (x < y) do
  begin
    dot(xCenter + x, yCenter + y);
    dot(xCenter + x, yCenter - y);
    dot(xCenter - x, yCenter + y);
    dot(xCenter - x, yCenter - y);
    dot(xCenter + y, yCenter + x);
    dot(xCenter + y, yCenter - x);
    dot(xCenter - y, yCenter + x);
    dot(xCenter - y, yCenter - x);
    inc(x);
    y := round(sqrt(r2 - x * x) + 0.5);
  end;
  if (x = y) then
  begin
    dot(xCenter + x, yCenter + y);
    dot(xCenter + x, yCenter - y);
    dot(xCenter - x, yCenter + y);
    dot(xCenter - x, yCenter - y);
  end;
end;

//draws a filled circle

procedure fillcircle(cx, cy, r: integer);
var
  x, y, r2, dx: integer;
begin
  if r = 0 then
  begin
    r := 1;
  end;
  r2 := r * r;
  for x := r downto 0 do
  begin
    y := round(sqrt(r2 - x * x));
    dx := cx - x;
    line(dx - 1, cy - y, dx - 1, cy + y);
    dx := cx + x;
    line(dx, cy - y, dx, cy + y);
  end;
end;

//draw an ellipse

procedure Ellipse(CX, CY, XRadius, YRadius: longint);
var
  X, Y: longint;
  XChange, YChange: longint;
  EllipseError: longint;
  TwoASquare, TwoBSquare: longint;
  StoppingX, StoppingY: longint;
begin
  if XRadius = YRadius then
  begin
    inc(YRadius);
  end;
  TwoASquare := 2 * XRadius * XRadius;
  TwoBSquare := 2 * YRadius * YRadius;
  X := XRadius;
  Y := 0;
  XChange := YRadius * YRadius * (1 - 2 * XRadius);
  YChange := XRadius * XRadius;
  EllipseError := 0;
  StoppingX := TwoBSquare * XRadius;
  StoppingY := 0;
  while (StoppingX >= StoppingY) do
  begin
    dot(CX + X, CY + Y);
    dot(CX - X, CY + Y);
    dot(CX - X, CY - Y);
    dot(CX + X, CY - Y);
    inc(Y);
    inc(StoppingY, TwoASquare);
    inc(EllipseError, YChange);
    inc(YChange, TwoASquare);
    if ((2 * EllipseError + XChange) > 0) then
    begin
      dec(X);
      dec(StoppingX, TwoBSquare);
      inc(EllipseError, XChange);
      inc(XChange, TwoBSquare)
    end;
  end;
  X := 0;
  Y := YRadius;
  XChange := YRadius * YRadius;
  YChange := XRadius * XRadius * (1 - 2 * YRadius);
  EllipseError := 0;
  StoppingX := 0;
  StoppingY := TwoASquare * YRadius;
  while (StoppingX <= StoppingY) do
  begin
    dot(CX + X, CY + Y);
    dot(CX - X, CY + Y);
    dot(CX - X, CY - Y);
    dot(CX + X, CY - Y);
    inc(X);
    inc(StoppingX, TwoBSquare);
    inc(EllipseError, XChange);
    inc(XChange, TwoBSquare);
    if ((2 * EllipseError + YChange) > 0) then
    begin
      dec(Y);
      dec(StoppingY, TwoASquare);
      inc(EllipseError, YChange);
      inc(YChange, TwoASquare)
    end;
  end;
end;

//draws a filled ellipse - warning it's slow

procedure fillellipse(xc, yc, width, height: integer);
var
  x, y: integer;
begin
  for y := -height to height do
  begin
    for x := -width to width do
    begin
      if (x * x * height * height + y * y * width * width <= height * height * width * width) then
      begin
        dot(xc + x, yc + y);
      end;
    end;
  end;
end;

//draws a triangle

procedure triangle(x1, y1, x2, y2, x3, y3: integer);
begin
  line(x1, y1, x2, y2);
  line(x2, y2, x3, y3);
  line(x3, y3, x1, y1);
end;

//draws a filled triangle - old TP code

procedure filltriangle(xa, ya, xb, yb, xc, yc: integer);
var
  y1, y2, y3, x1, x2, x3: longint;
  dx12, dx13, dx23: longint;
  dy12, dy13, dy23, dy: longint;
  a, b: longint;
begin
  if ya = yb then
  begin
    inc(yb);
  end;
  if ya = yc then
  begin
    inc(yc);
  end;
  if yc = yb then
  begin
    inc(yb);
  end;
  if (ya <> yb) and (ya <> yc) and (yc <> yb) then
  begin
    if (ya > yb) and (ya > yc) then
    begin
      y1 := ya;
      x1 := xa;
      if yb > yc then
      begin
        y2 := yb;
        x2 := xb;
        y3 := yc;
        x3 := xc;
      end
      else
      begin
        y2 := yc;
        x2 := xc;
        y3 := yb;
        x3 := xb;
      end;
    end
    else if (yb > ya) and (yb > yc) then
    begin
      y1 := yb;
      x1 := xb;
      if ya > yc then
      begin
        y2 := ya;
        x2 := xa;
        y3 := yc;
        x3 := xc;
      end
      else
      begin
        y2 := yc;
        x2 := xc;
        y3 := ya;
        x3 := xa;
      end;
    end
    else if (yc > yb) and (yc > ya) then
    begin
      y1 := yc;
      x1 := xc;
      if yb >= ya then
      begin
        y2 := yb;
        x2 := xb;
        y3 := ya;
        x3 := xa;
      end
      else
      begin
        y2 := ya;
        x2 := xa;
        y3 := yb;
        x3 := xb;
      end;
    end;
    dx12 := x2 - x1;
    dy12 := y2 - y1;
    dx23 := x3 - x2;
    dy23 := y3 - y2;
    dx13 := x3 - x1;
    dy13 := y3 - y1;
    a := x2 - ((y2 - y3 + dy23) * dx23) div dy23;
    b := x3 + (-dy23 * dx13) div (dy13);
    if (a < b) then
    begin
      line(a, y2, b, y2);
      // for x:=a to b do dot (x,y2);
      for dy := 0 to -dy23 - 1 do
      begin
        a := x2 + ((dy23 + dy) * dx23) div dy23;
        b := x3 + (dy * dx13) div (dy13);
        line(a, dy + y3, b, dy + y3);
        // for x:=a to b do dot (x,dy+y3);
      end;
      for dy := -dy23 + 1 to -dy13 do
      begin
        a := x2 + ((dy23 + dy) * dx12) div dy12;
        b := x3 + (dy * dx13) div (dy13);
        line(a, dy + y3, b, dy + y3);
        //for x:=a to b do dot (x,dy+y3);
      end;
    end
    else
    begin
      //for x:=b to a do dot (x,y2);
      line(b, y2, a, y2);
      for dy := 0 to -dy23 - 1 do
      begin
        a := x2 + ((dy23 + dy) * dx23) div dy23;
        b := x3 + (dy * dx13) div (dy13);
        line(a, dy + y3, b, dy + y3);
        // for x:=b to a do dot (x,dy+y3);
      end;
      for dy := -dy23 + 1 to -dy13 do
      begin
        a := x2 + ((dy23 + dy) * dx12) div dy12;
        b := x3 + (dy * dx13) div (dy13);
        line(a, dy + y3, b, dy + y3);
        //for x:=b to a do dot(x,dy+y3);
      end;
    end;
  end;

end;

//clears the active window

procedure clearwindow;
begin
  color(bkr, bkg, bkb, bka);
  SDL_RenderClear(SDL_GetRenderer(SDL_GetWindowFromID(activewindow)));
end;

procedure backcolor(red, green, blue, alpha: integer);
begin
  bkr := red;
  bkg := green;
  bkb := blue;
  bka := alpha;
end;

procedure togglefullscreen;
begin
  if fullscreenmode = false then
  begin
    SDL_SetWindowFullscreen(SDL_GetWindowFromID(activewindow), SDL_WINDOW_FULLSCREEN_DESKTOP);
    fullscreenmode := true;
  end
  else
  begin
    SDL_SetWindowFullscreen(SDL_GetWindowFromID(activewindow), 0);
    fullscreenmode := false;
  end;

end;

procedure color(red, green, blue, alpha: integer);
begin
  SDL_SetRenderDrawBlendMode(SDL_GetRenderer(SDL_GetWindowFromID(activewindow)), SDL_BLENDMODE_BLEND);
  SDL_SetRenderDrawColor(SDL_GetRenderer(SDL_GetWindowFromID(activewindow)), red, green, blue, alpha);

end;

procedure spritecolor(tex: PSDL_Texture; red, green, blue, alpha: integer);
begin
  SDL_SetTextureColorMod(tex, red, green, blue);
  SDL_SetTextureAlphaMod(tex, alpha);
end;

procedure redraw;
begin
  SDL_RenderPresent(SDL_GetRenderer(SDL_GetWindowFromID(activewindow)));
end;

procedure wait(ms: integer);
begin
  SDL_Delay(ms);
end;

function loadimage(filename: string): PSDL_Surface;
var
  image: PSDL_Surface;
  color: integer;
begin
  if fileexists(filename) then
  begin

    image := IMG_LOAD(PCHar(filename));
  end
  else
  begin
    SDL_ShowSimpleMessageBox(SDL_MESSAGEBOX_ERROR, 'File not found', PCHAR('Can''t find image file: ' + filename), nil);
    SDL_Quit();
    Halt;
  end;
  if (image = nil) then
  begin
    Messagebox(1, 'Error', 'SDL_LoadBMP Error: ' + SDL_GetError);
    SDL_Quit;
    Halt;
  end;
  if colorkey = true then
  begin
    color := SDL_MapRGB(image.format, ckr, ckg, ckb);
    SDL_SetColorKey(image, 1, color);
  end;
  result := image;
end;

procedure freeimage(srcimage: PSDL_Surface);
begin
  SDL_Freesurface(srcimage);
end;

procedure freesprite(srcsprite: PSDL_Texture);
begin
  SDL_DestroyTexture(srcsprite);
end;

procedure closeapplication;
begin
{$IFDEF useaudio}
  MIX_HALTMUSIC;
  MIX_CLOSEAUDIO;
{$ENDIF}
  if SDL_JoystickGetAttached(joystick) = SDL_TRUE then
  begin
    SDL_JoystickClose(joystick);
  end;
  IMG_Quit();
  SDL_Quit();
end;

procedure Messagebox(typus: integer; title, textmessage: string);
begin
  SDL_ShowSimpleMessageBox(typus, Pchar(title), PChar(textmessage), nil);
end;

procedure textsize(size: integer);
begin
  fontsize := size;
end;

procedure texttype(typus: integer);
begin
  if (typus >= 1) and (typus <= 2) then
  begin
    fonttype := typus;
  end
  else
  begin
    fonttype := 1;
  end;
end;

procedure textinputbackcolor(red, green, blue, alpha: integer);
begin
  tbred := red;
  tbgreen := green;
  tbblue := blue;
  tbalpha := alpha;
end;

procedure textinputcolor(red, green, blue, alpha: integer);
begin
  tcred := red;
  tcgreen := green;
  tcblue := blue;
  tcalpha := alpha;
end;

function textwidth(TextString: string): integer;
var
  len: integer;
begin
  len := length(TextString);
  result := len * 8 * fontsize;
end;

function textheight(): integer;
begin
  result := fontsize * 8;
end;

procedure drawchar(x, y: integer; ch: char);
var
  horiz, vert, oldx: integer;
begin
  oldx := x;
  for vert := 0 to 7 do
  begin
    for horiz := 0 to 7 do
    begin
      case fonttype of
        1:
          begin
            if (DefaultFontData[ch][vert][horiz]) = 1 then
            begin
              fillrectangle(x, y, fontsize, fontsize);
            end;
          end;
        2:
          begin
            if (SerifFontData[ch][vert][horiz]) = 1 then
            begin
              fillrectangle(x, y, fontsize, fontsize);
            end;
          end;
      end;
      x := x + fontsize;
    end;
    y := y + fontsize;
    x := oldx;
  end;
end;

procedure drawtext(str: string; x, y: integer);
var
  ch: char;
  i, len: integer;

begin

  len := length(str);
  for i := 1 to len do
  begin
    ch := str[i];
    drawchar(x, y, ch);
    x := x + fontsize * 8;
  end;
end;

//now it's getting complicated, although SDL2 has got a text input function - anyway it's shorter than it was in EGSL

function input(x, y, maxlen: integer): string;
var
  done: boolean;
  mytext: string;
  event: TSDL_event;
begin
  SDL_SetTextInputRect(nil);
  SDL_StartTextInput();
  mytext := '';
  color(tcred, tcgreen, tcblue, tcalpha);
  fillrectangle(x + textwidth(mytext), y, textheight(), textheight());
  sync;
  repeat

    done := false;

    //color (tcred,tcgreen,tcblue,tcalpha);
    //drawtext (x,y,mytext);
    //fillrectangle (x+textwidth (mytext),y,textheight(),textheight());

    if (SDL_PollEvent(@event) <> 0) then
    begin
      if (event.type_) = SDL_TEXTINPUT then
      begin
        sync;
        mytext := mytext + event.text.text;
        if length(mytext) > maxlen then
        begin
          sync;
          mytext := leftstr(mytext, length(mytext) - 1);
          color(tbred, tbgreen, tbblue, tbalpha);
          fillrectangle(x, y, textwidth(mytext) + textheight(), textheight());

          color(tcred, tcgreen, tcblue, tcalpha);
          drawtext(mytext, x, y);
          fillrectangle(x + textwidth(mytext), y, textheight(), textheight());
          sync;
        end;
        color(tbred, tbgreen, tbblue, tbalpha);
        fillrectangle(x, y, textwidth(mytext), textheight());

        color(tcred, tcgreen, tcblue, tcalpha);
        drawtext(mytext, x, y);
        fillrectangle(x + textwidth(mytext), y, textheight(), textheight());
        sync;
      end;
      if (event.type_) = SDL_TEXTEDITING then
      begin
        mytext := event.edit.text;
        sync;
      end;
      if (event.type_ = SDL_QUITEV) then
      begin
        SDL_Quit();
        halt;
      end;
      if (event.type_) = SDL_KEYDOWN then
      begin
        if ((event.key.keysym.scancode) = SDL_SCANCODE_RETURN) or ((event.key.keysym.scancode) = SDL_SCANCODE_KP_ENTER)
          or ((event.key.keysym.scancode) = SDL_SCANCODE_ESCAPE) then
        begin
          done := true;
        end;
        if (event.key.keysym.scancode) = SDL_SCANCODE_BACKSPACE then
        begin
          sync;
          mytext := leftstr(mytext, length(mytext) - 1);
          color(tbred, tbgreen, tbblue, tbalpha);
          fillrectangle(x, y, textwidth(mytext) + textheight() * 2, textheight());
          color(tcred, tcgreen, tcblue, tcalpha);
          drawtext(mytext, x, y);
          fillrectangle(x + textwidth(mytext), y, textheight(), textheight());
          sync;
        end;
      end;

    end;

  until done = true;
  SDL_StopTextInput();
  color(tbred, tbgreen, tbblue, tbalpha);
  fillrectangle(x, y, textwidth(mytext) + textheight(), textheight());
  color(tcred, tcgreen, tcblue, tcalpha);
  drawtext(mytext, x, y);
  result := mytext;
end;

//now it's getting complicated, although SDL2 has got a text input function - anyway it's shorter than it was in EGSL

function bmpinput(fontnumber, x, y, maxlen: integer): string;
var
  done: boolean;
  mytext: string;
  event: TSDL_event;
begin
  if fontnumber < bmpfont.index then
  begin

    SDL_SetTextInputRect(nil);
    SDL_StartTextInput();
    mytext := '';
    color(bmpfont.inpred[fontnumber], bmpfont.inpgreen[fontnumber], bmpfont.inpblue[fontnumber],
      bmpfont.inpalpha[fontnumber]);
    fillrectangle(x + length(mytext) * bmpfont.width[fontnumber] * bmpfont.xfactor[fontnumber], y,
      bmpfont.width[fontnumber] * bmpfont.xfactor[fontnumber], bmpfont.height[fontnumber] * bmpfont.yfactor[fontnumber]);
    sync;
    repeat

      done := false;

      //color (tcred,tcgreen,tcblue,tcalpha);
      //drawtext (x,y,mytext);
      //fillrectangle (x+textwidth (mytext),y,textheight(),textheight());

      if (SDL_PollEvent(@event) <> 0) then
      begin
        if (event.type_) = SDL_TEXTINPUT then
        begin
          sync;
          mytext := mytext + event.text.text;
          if length(mytext) > maxlen then
          begin
            sync;
            mytext := leftstr(mytext, length(mytext) - 1);
            color(bmpfont.binpred[fontnumber], bmpfont.binpgreen[fontnumber], bmpfont.binpblue[fontnumber],
              bmpfont.binpalpha[fontnumber]);
            fillrectangle(x, y, (length(mytext) * bmpfont.width[fontnumber] * bmpfont.xfactor[fontnumber]) +
              bmpfont.width[fontnumber] * bmpfont.xfactor[fontnumber], bmpfont.height[fontnumber] *
              bmpfont.yfactor[fontnumber]);

            color(bmpfont.inpred[fontnumber], bmpfont.inpgreen[fontnumber], bmpfont.inpblue[fontnumber],
              bmpfont.inpalpha[fontnumber]);
            bmptext(mytext, x, y, fontnumber);
            fillrectangle(x + length(mytext) * bmpfont.width[fontnumber] * bmpfont.xfactor[fontnumber], y,
              bmpfont.width[fontnumber] * bmpfont.xfactor[fontnumber], bmpfont.height[fontnumber] *
              bmpfont.yfactor[fontnumber]);
            sync;
          end;
          color(bmpfont.binpred[fontnumber], bmpfont.binpgreen[fontnumber], bmpfont.binpblue[fontnumber],
            bmpfont.binpalpha[fontnumber]);
          fillrectangle(x, y, length(mytext) * bmpfont.width[fontnumber] * bmpfont.xfactor[fontnumber],
            bmpfont.height[fontnumber] * bmpfont.yfactor[fontnumber]);

          color(bmpfont.inpred[fontnumber], bmpfont.inpgreen[fontnumber], bmpfont.inpblue[fontnumber],
            bmpfont.inpalpha[fontnumber]);

          bmptext(mytext, x, y, fontnumber);
          fillrectangle(x + length(mytext) * bmpfont.width[fontnumber] * bmpfont.xfactor[fontnumber], y,
            bmpfont.width[fontnumber] * bmpfont.xfactor[fontnumber], bmpfont.height[fontnumber] *
            bmpfont.yfactor[fontnumber]);
          sync;
        end;
        if (event.type_) = SDL_TEXTEDITING then
        begin
          mytext := event.edit.text;
          sync;
        end;
        if (event.type_ = SDL_QUITEV) then
        begin
          SDL_Quit();
          halt;
        end;
        if (event.type_) = SDL_KEYDOWN then
        begin
          if ((event.key.keysym.scancode) = SDL_SCANCODE_RETURN) or ((event.key.keysym.scancode) = SDL_SCANCODE_KP_ENTER)
            or ((event.key.keysym.scancode) = SDL_SCANCODE_ESCAPE) then
          begin
            done := true;
          end;
          if (event.key.keysym.scancode) = SDL_SCANCODE_BACKSPACE then
          begin
            sync;
            mytext := leftstr(mytext, length(mytext) - 1);
            color(bmpfont.binpred[fontnumber], bmpfont.binpgreen[fontnumber], bmpfont.binpblue[fontnumber],
              bmpfont.binpalpha[fontnumber]);
            fillrectangle(x, y, (length(mytext) * bmpfont.width[fontnumber] * bmpfont.xfactor[fontnumber]) +
              (bmpfont.width[fontnumber] * bmpfont.xfactor[fontnumber] * 2), bmpfont.height[fontnumber] *
              bmpfont.yfactor[fontnumber]);

            color(bmpfont.inpred[fontnumber], bmpfont.inpgreen[fontnumber], bmpfont.inpblue[fontnumber],
              bmpfont.inpalpha[fontnumber]);

            bmptext(mytext, x, y, fontnumber);
            fillrectangle(x + length(mytext) * bmpfont.width[fontnumber] * bmpfont.xfactor[fontnumber], y,
              bmpfont.width[fontnumber] * bmpfont.xfactor[fontnumber], bmpfont.height[fontnumber] *
              bmpfont.yfactor[fontnumber]);
            sync;
          end;
        end;

      end;

    until done = true;
    color(bmpfont.binpred[fontnumber], bmpfont.binpgreen[fontnumber], bmpfont.binpblue[fontnumber],
      bmpfont.binpalpha[fontnumber]);
    fillrectangle(x, y, (length(mytext) * bmpfont.width[fontnumber] * bmpfont.xfactor[fontnumber]) +
      (bmpfont.width[fontnumber] * bmpfont.xfactor[fontnumber]), bmpfont.height[fontnumber] *
      bmpfont.yfactor[fontnumber]);
    bmptext(mytext, x, y, fontnumber);
    sync;
    SDL_StopTextInput();
    result := mytext;
  end
  else
  begin
    SDL_ShowSimpleMessageBox(SDL_MESSAGEBOX_ERROR, 'No font', PCHAR('Font not loaded!'), nil);
    SDL_Quit();
    Halt;
  end;
end;

function bmpfontwidth(fontnumber: integer): integer;
begin
  if fontnumber < bmpfont.index then
  begin
    result := bmpfont.width[fontnumber] * bmpfont.xfactor[fontnumber];
  end
  else
  begin
    SDL_ShowSimpleMessageBox(SDL_MESSAGEBOX_ERROR, 'No font', PCHAR('Font not loaded!'), nil);
    SDL_Quit();
    Halt;
  end;
end;

function bmpfontheight(fontnumber: integer): integer;
begin
  if fontnumber < bmpfont.index then
  begin
    result := bmpfont.height[fontnumber] * bmpfont.yfactor[fontnumber];
  end
  else
  begin
    SDL_ShowSimpleMessageBox(SDL_MESSAGEBOX_ERROR, 'No font', PCHAR('Font not loaded!'), nil);
    SDL_Quit();
    Halt;
  end;
end;

function fullscreen: boolean;
begin
  if fullscreenmode = true then
  begin
    result := true
  end
  else
  begin
    result := false;
  end;

end;

//returns ms how long the program is running

function timerticks: integer;
begin
  Result := SDL_GetTicks;
end;

procedure SetFrameTimer(frames: integer);
begin
  if frames = 0 then
    frames := 1;
  TICK_INTERVAL := 1000 div frames;
end;

function TimeLeft: UInt32;
var
  now: UInt32;
begin
  now := SDL_GetTicks;
  if next_time <= now then
  begin
    next_time := now + TICK_INTERVAL;
    result := 0;
    exit;
  end;
  result := next_time - now;
end;

procedure sync();
begin
  redraw;
  wait(timeleft());
end;

procedure bmpinputbackcolor(fontnumber, red, green, blue, alpha: integer);
begin
  if fontnumber < bmpfont.index then
  begin
    bmpfont.binpred[fontnumber] := red;
    bmpfont.binpgreen[fontnumber] := green;
    bmpfont.binpblue[fontnumber] := blue;
    bmpfont.binpalpha[fontnumber] := alpha;
  end
  else
  begin
    SDL_ShowSimpleMessageBox(SDL_MESSAGEBOX_ERROR, 'No font', PCHAR('Font not loaded!'), nil);
    SDL_Quit();
    Halt;
  end;
end;

procedure bmpinputcolor(fontnumber, red, green, blue, alpha: integer);
begin
  if fontnumber < bmpfont.index then
  begin
    bmpfont.inpred[fontnumber] := red;
    bmpfont.inpgreen[fontnumber] := green;
    bmpfont.inpblue[fontnumber] := blue;
    bmpfont.inpalpha[fontnumber] := alpha;
  end
  else
  begin
    SDL_ShowSimpleMessageBox(SDL_MESSAGEBOX_ERROR, 'No font', PCHAR('Font not loaded!'), nil);
    SDL_Quit();
    Halt;
  end;
end;

function createbmpfont(sprite: p2dsprite; width, height: integer; fontface: ansistring): integer;
begin

  inc(bmpfont.index);
  SetLength(bmpfont.image, bmpfont.index);
  SetLength(bmpfont.width, bmpfont.index);
  SetLength(bmpfont.height, bmpfont.index);
  SetLength(bmpfont.face, bmpfont.index);
  SetLength(bmpfont.lines, bmpfont.index);
  SetLength(bmpfont.rows, bmpfont.index);
  SetLength(bmpfont.xfactor, bmpfont.index);
  SetLength(bmpfont.yfactor, bmpfont.index);
  SetLength(bmpfont.angle, bmpfont.index);
  SetLength(bmpfont.inpred, bmpfont.index);
  SetLength(bmpfont.inpgreen, bmpfont.index);
  SetLength(bmpfont.inpblue, bmpfont.index);
  SetLength(bmpfont.inpalpha, bmpfont.index);
  SetLength(bmpfont.binpred, bmpfont.index);
  SetLength(bmpfont.binpgreen, bmpfont.index);
  SetLength(bmpfont.binpblue, bmpfont.index);
  SetLength(bmpfont.binpalpha, bmpfont.index);

  bmpfont.image[bmpfont.index - 1] := sprite;
  bmpfont.width[bmpfont.index - 1] := width;
  bmpfont.height[bmpfont.index - 1] := height;
  bmpfont.lines[bmpfont.index - 1] := round(spriteheight(bmpfont.image[bmpfont.index - 1]) / height);
  bmpfont.rows[bmpfont.index - 1] := round(spritewidth(bmpfont.image[bmpfont.index - 1]) / width);
  bmpfont.face[bmpfont.index - 1] := fontface;
  bmpfont.xfactor[bmpfont.index - 1] := 1;
  bmpfont.yfactor[bmpfont.index - 1] := 1;
  bmpfont.inpred[bmpfont.index - 1] := 255;
  bmpfont.inpgreen[bmpfont.index - 1] := 255;
  bmpfont.inpblue[bmpfont.index - 1] := 255;
  bmpfont.inpalpha[bmpfont.index - 1] := 255;

  bmpfont.binpred[bmpfont.index - 1] := 0;
  bmpfont.binpgreen[bmpfont.index - 1] := 0;
  bmpfont.binpblue[bmpfont.index - 1] := 0;
  bmpfont.binpalpha[bmpfont.index - 1] := 255;

  bmpfont.angle[bmpfont.index - 1] := 0;
  result := bmpfont.index - 1;
end;

procedure bmpfontsize(fontnumber: integer; xf, yf: integer);
begin
  if fontnumber < bmpfont.index then
  begin
    bmpfont.xfactor[fontnumber] := xf;
    bmpfont.yfactor[fontnumber] := yf;
  end
  else
  begin
    SDL_ShowSimpleMessageBox(SDL_MESSAGEBOX_ERROR, 'No font', PCHAR('Font not loaded!'), nil);
    SDL_Quit();
    Halt;
  end;
end;

procedure bmpfontangle(fontnumber: integer; angle: real);
begin
  if fontnumber < bmpfont.index then
  begin
    bmpfont.angle[fontnumber] := angle;
  end
  else
  begin
    SDL_ShowSimpleMessageBox(SDL_MESSAGEBOX_ERROR, 'No font', PCHAR('Font not loaded!'), nil);
    SDL_Quit();
    Halt;
  end;
end;

procedure bmptext(textstr: ansistring; xspace, yspace: integer; fontnumber: integer);
var
  counter, textlength, chars, i, xs, ys, xfont, yfont: integer;
  mychar: string;
  space: boolean;
begin
  if fontnumber < bmpfont.index then
  begin
    chars := length(bmpfont.face[fontnumber]);
    textlength := length(textstr);
    xfont := 0;
    yfont := 0;
    for i := 1 to textlength do
    begin
      mychar := copy(textstr, i, 1);
      counter := 1;
      xs := 1;
      ys := 1;
      repeat
        space := false;
        if (copy(bmpfont.face[fontnumber], counter, 1) = mychar) then
        begin
          xfont := xs;
          yfont := ys;
        end
        else if (mychar = ' ') then
        begin
          space := true;
        end;
        xs := xs + 1;
        if xs > bmpfont.rows[fontnumber] then
        begin
          ys := ys + 1;
          xs := 1;
        end;
        counter := counter + 1;
      until counter > chars;
      if space = false then
      begin
        // drawspritepart (sprite:PSDL_Texture; x,y,startx,starty,xoffset,yoffset:integer;width,height,angle:real;vflip,hflip:boolean);
        drawspritepart(bmpfont.image[fontnumber], xspace, yspace, (xfont - 1) * bmpfont.width[fontnumber], (yfont - 1) *
          bmpfont.height[fontnumber], bmpfont.width[fontnumber], bmpfont.height[fontnumber], bmpfont.xfactor[fontnumber],
          bmpfont.yfactor[fontnumber], bmpfont.angle[fontnumber], false, false);
      end;
      xspace := xspace + bmpfont.width[fontnumber] * bmpfont.xfactor[fontnumber];
      space := false;
    end;
  end
  else
  begin
    SDL_ShowSimpleMessageBox(SDL_MESSAGEBOX_ERROR, 'No font', PCHAR('Font not loaded!'), nil);
    SDL_Quit();
    Halt;
  end;
end;

begin
  if (SDL_Init(SDL_INIT_EVERYTHING) <> 0) then
  begin
    WriteLn('SDL_Init Error: ', SDL_GetError);
    SDL_Quit;
    Exit;
  end;
  if (SDL_NumJoysticks > 0) then
  begin
    joystick := SDL_JoystickOpen(0);
  end;
  bkr := 0;
  bkg := 0;
  bkb := 0;
  bka := 255;
  colorkey := true;
  ckr := 255;
  ckg := 0;
  ckb := 255;
  fontsize := 1;
  fonttype := 1;
  tbred := 0;
  tbgreen := 0;
  tbblue := 0;
  tbalpha := 255;
  tcred := 255;
  tcgreen := 255;
  tcblue := 255;
  tcalpha := 255;
  fullscreenmode := false;
  virtualsize := false;
  bmpfont.index := 0;
end.
