{
   p2dcollision.pas

   Copyright 2013-2015 Markus Mangold <info@retrogamecoding.org>


  This software is provided 'as-is', without any express or implied
  warranty. In no event will the authors be held liable for any damages
  arising from the use of this software.

  Permission is granted to anyone to use this software for any purpose,
  including commercial applications, and to alter it and redistribute it
  freely, subject to the following restrictions:

   1. The origin of this software must not be misrepresented; you must not
   claim that you wrote the original software. If you use this software
   in a product, an acknowledgment in the product documentation would be
   appreciated but is not required.

   2. Altered source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

   3. This notice may not be removed or altered from any source
   distribution.

}

unit P2DCollision;

interface

uses
  SysUtils,
  SDL2,
  P2DVideo;

function Boxcoll(x1, y1, w1, h1, x2, y2, w2, h2: integer): boolean;
function CircleColl(x1, y1, r1, x2, y2, r2, offset: integer): boolean;
function sidecoll(x, y, w, h, dx, dy, rx, ry, rw, rh: integer): ansistring;
function InsideTriangle(px, py, x1, y1, x2, y2, x3, y3: integer): boolean;
function InsideCircle(px, py, cx, cy, radius: integer): boolean;
function InsideRectangle(posx, posy, x1, y1, x2, y2: integer): boolean;
function ImageColl(ImageA: p2dsprite; ax, ay: integer; ImageB: p2dsprite; bx, by: integer): boolean;
function RoundColl(aSurface: p2dsprite; x1, y1: integer; bSurface: p2dsprite; x2, y2, offset: integer): boolean;
function circletoboxcoll(cx, cy, r, x1, y1, w, h: integer): boolean;
function Intersect(x1, y1, x2, y2, x3, y3, x4, y4: Double): Boolean;
function LineHitsCircle(LineX1, LineY1, LineX2, LineY2, CircleX, CircleY, CircleRadius: double): boolean;

implementation

function Boxcoll(x1, y1, w1, h1, x2, y2, w2, h2: integer): boolean;
begin
  Result := false;
  if x2 + w2 < x1 then
    exit; //just checking if their
  if x2 > x1 + w1 then
    exit; //bounding boxes even touch

  if y2 + h2 < y1 then
    exit;
  if y2 > y1 + h1 then
    exit;

  Result := true; //bounding boxes intersect
end;

function CircleColl(x1, y1, r1, x2, y2, r2, offset: integer): boolean;
var
  xdiff, ydiff, dcentre_sq, r_sum_sq: integer;
begin
  xdiff := x2 - x1; // x plane difference
  ydiff := y2 - y1; // y plane difference

  { distance between the circles centres squared }
  dcentre_sq := ydiff * ydiff + xdiff * xdiff;

  { calculate sum of radiuses squared }
  r_sum_sq := r1 + r2; // square on seperate line, so
  r_sum_sq := r_sum_sq * r_sum_sq; // dont recompute r1 + r2

  Result := (dcentre_sq - r_sum_sq <= (offset * offset));
end;

function RoundColl(aSurface: p2dsprite; x1, y1: integer; bSurface: p2dsprite; x2, y2, offset: integer): boolean;
var
  r1, r2: integer;
begin

  { if radius is not specified
  we approximate them using p2sprite's
  width and height average and divide by 2 }

  r1 := (spritewidth(aSurface) + spriteheight(aSurface)) div 4; // same as / 2) / 2;
  r2 := (spritewidth(bSurface) + spriteheight(bSurface)) div 4;

  x1 := x1 + spritewidth(aSurface) div 2; // offset x and y
  y1 := y1 + spriteheight(aSurface) div 2; // co-ordinates into
  // centre of image
  x2 := x2 + spritewidth(bSurface) div 2;
  y2 := y2 + spriteheight(bSurface) div 2;

  Result := CircleColl(x1, y1, r1, x2, y2, r2, offset);
end;

function sidecoll(x, y, w, h, dx, dy, rx, ry, rw, rh: integer): ansistring;
begin
  result := 'none';
  if (dx = 0) and (dy = 0) then
  begin
    result := 'none';
    exit;
  end;
  if dx < 0 then
  begin
    if x >= rx + rw then
    begin
      if x + dx < rx + rw then
      begin
        if (y + dy < ry + rh) and (ry < y + dy + h) then
        begin
          result := 'left';
          exit;
        end;
      end;
    end;
  end
  else if dx > 0 then
  begin
    if rx >= x + w then
    begin
      if rx < x + w + dx then
      begin
        if (y + dy < ry + rh) and (ry < y + dy + h) then
        begin
          result := 'right';
          exit;
        end;
      end;
    end;
  end;

  if dy < 0 then
  begin
    if y >= ry + rh then
    begin
      if y + dy < ry + rh then
      begin
        if (x < rx + rw) and (rx < x + w) then
        begin
          result := 'top';
          exit;
        end;
      end;
    end;
  end
  else if dy > 0 then
  begin
    if ry >= y + h then
    begin
      if ry < y + h + dy then
      begin
        if (x < rx + rw) and (rx < x + w) then
        begin
          result := 'bottom';
          exit;
        end;
      end;
    end;
  end;
end;

function ImageColl(ImageA: p2dsprite; ax, ay: integer; ImageB: p2dsprite; bx, by: integer): boolean;
begin

  if boxcoll(ax, ay, spritewidth(ImageA), spriteheight(ImageA), bx, by, spritewidth(ImageB), spriteheight(ImageB)) = true
    then
    result := true
  else
    result := false;
end;

function circletoboxcoll(cx, cy, r, x1, y1, w, h: integer): boolean;
var
  testx, testy, x2, y2: integer;
begin
  testx := cx;
  testy := cy;
  x2 := x1 + w;
  y2 := y1 + h;
  if testx < x1 then
  begin
    testx := x1;
  end;
  if testx > x2 then
  begin
    testx := x2;
  end;
  if testy < y1 then
  begin
    testy := y1;
  end;
  if testy > y2 then
  begin
    testy := y2;
  end;
  Result := (cx - testx) * (cx - testx) + (cy - testy) * (cy - testy) < r * r;
end;

function Orientation(x1, y1, x2, y2, Px, Py: Double): Integer;
var
  Orin: Double;
begin
  (* Linear determinant of the 3 points *)
  Orin := (x2 - x1) * (py - y1) - (px - x1) * (y2 - y1);

  if Orin > 0.0 then
    Result := +1 (* Orientaion is to the right-hand side  *)
  else if Orin < 0.0 then
    Result := -1 (* Orientaion is to the left-hand side   *)
  else
    Result := 0; (* Orientaion is neutral if result is 0  *)
end;

function Intersect(x1, y1, x2, y2, x3, y3, x4, y4: Double): Boolean;
begin
  Result := (Orientation(x1, y1, x2, y2, x3, y3) <> Orientation(x1, y1, x2, y2, x4, y4))
    and
    (Orientation(x3, y3, x4, y4, x1, y1) <> Orientation(x3, y3, x4, y4, x2, y2));
end;
(* End Of SegmentIntersect *)

function getOrientationResult(x1, y1, x2, y2, px, py: integer): integer;
var
  orientation: integer;
begin
  orientation := ((x2 - x1) * (py - y1)) - ((px - x1) * (y2 - y1));
  if orientation > 0 then
  begin
    result := 1;
  end
  else if orientation < 0 then
  begin
    result := -1;
  end
  else
  begin
    result := 0;
  end;
end;

function InsideTriangle(px, py, x1, y1, x2, y2, x3, y3: integer): boolean;
var
  o1, o2, o3: integer;
begin
  o1 := getOrientationResult(x1, y1, x2, y2, px, py);
  o2 := getOrientationResult(x2, y2, x3, y3, px, py);
  o3 := getOrientationResult(x3, y3, x1, y1, px, py);
  result := ((o1 = o2) and (o2 = o3));
end;

function InsideCircle(px, py, cx, cy, radius: integer): boolean;
var
  distance: real;
begin
  distance := sqrt((px - cx) * (px - cx) + (py - cy) * (py - cy));
  if distance < radius then
  begin
    result := true;
  end
  else if distance > radius then
  begin
    result := false;
  end;
end;

function InsideRectangle(posx, posy, x1, y1, x2, y2: integer): boolean;
begin
  if (posx > x1) and (posx < x2) and (posy > y1) and (posy < y2) then
  begin
    result := true;
  end
  else
  begin
    result := false;
  end;
end;

function LineHitsCircle(LineX1, LineY1, LineX2, LineY2, CircleX, CircleY, CircleRadius: double): boolean;
var
  x1, x2, x: double;
  d: double;
  d_helper: double;
  m, n: double;
  outpos: double;
  tmp: double;
begin
  LineX1 := LineX1 - CircleX;
  LineY1 := LineY1 - CircleY;
  LineX2 := LineX2 - CircleX;
  LineY2 := LineY2 - CircleY;

  if (LineX2 - LineX1 = 0) then
  begin
    if (LineY2 - LineY1 = 0) then
    begin

      result := sqr(LineX1) + sqr(LineY1) <= sqr(CircleRadius);
      exit;
    end
    else
    begin
      tmp := LineY1;
      LineY1 := LineX1;
      LineX1 := tmp;

      tmp := LineY2;
      LineY2 := LineX2;
      LineX2 := tmp;
    end;
  end;

  m := (LineY2 - LineY1) / (LineX2 - LineX1);
  n := LineY1 - m * LineX1;
  d_helper := 4 * sqr(m) * sqr(n) - 4 * (1 + sqr(m)) * (sqr(n) + 2 * n - sqr(CircleRadius));

  if d_helper < 0 then
  begin

    result := False;
  end
  else
  begin
    d := sqrt(d_helper);
    x1 := (-2 * m * n - d) / (2 + 2 * sqr(m));
    x2 := (-2 * m * n + d) / (2 + 2 * sqr(m));

    if (x1 >= LineX1) and ((x1 - LineX1) <= (x2 - LineX1)) then
      x := x1
    else
      x := x2;

    OutPos := abs((x - LineX1) / (LineX2 - LineX1));

    result := (OutPos >= 0.0) and (OutPos <= 1.0);
  end;
end;

begin

end.
