
(* 
 * This utility looks up the moves and their scores in a Polyglot book
 *
 * Usage:
 * pg_show <book> <hex key>
 *
 * You can find the hex key of a FEN using pg_key. 
 *
 * This code is released in the public domain by Michel Van den Bergh.
 *
 *)

(* Pascal version by R. Chastain. *) 

uses
  SysUtils;

type
  entry_t = record
    key: UInt64;
    move: UInt16;
    weight: UInt16;
    learn: UInt32;
  end;

const
  entry_none: entry_t = (
    key: 0;
    move: 0;
    weight: 0;
    learn: 0
  );
  promote_pieces = 'nbrq';
  MAX_MOVES = 100;

function int_from_file(const f: THandle; const l: integer; var r: UInt64): boolean;
var
  i: integer;
  c: byte;
begin
  for i := 0 to Pred(l) do
  begin
    if FileRead(f, c, SizeOf(c)) <> SizeOf(c) then
      Exit(FALSE);
    r := (r shl 8) + c;
  end;
  result := TRUE;
end;

function entry_from_file(const f: THandle; var entry: entry_t): boolean;
var
  r: UInt64 = 0;
begin
  if not int_from_file(f, 8, r) then Exit(FALSE);
  entry.key := r;
  if not int_from_file(f, 2, r) then Exit(FALSE);
  entry.move := UInt16(r);
  if not int_from_file(f, 2, r) then Exit(FALSE);
  entry.weight := UInt16(r);
  if not int_from_file(f, 4, r) then Exit(FALSE);
  entry.learn := UInt32(r);
  result := TRUE;
end;

function find_key(const f: THandle; const key: UInt64; var entry: entry_t): integer;
var
  first, last, middle: integer;
  first_entry, last_entry, middle_entry: entry_t;
begin
  first_entry := entry_none;
  first := -1;
  if FileSeek(f, -16, fsFromEnd) = -1 then
  begin
    entry := entry_none;
    entry.key := key + 1; // hack
    Exit(-1);
  end;
  last := FileSeek(f, 0, fsFromCurrent) div 16;
  entry_from_file(f, last_entry);
  while TRUE do
  begin
    if last - first = 1 then
    begin
      entry := last_entry;
      Exit(last);
    end;
    middle := (first + last) div 2;
    FileSeek(f, 16 * middle, fsFromBeginning);
    entry_from_file(f, middle_entry);
    if key <= middle_entry.key then
    begin
      last := middle;
      last_entry := middle_entry;
    end else
    begin
      first := middle;
      first_entry := middle_entry;
    end;
  end;
end;

procedure move_to_string(var move_s: string; const move: UInt16);
var
  f, fr, ff, t, tr, tf, p: integer;
begin
  f := (move shr 6) and 63;
  fr := (f shr 3) and 7;
  ff := f and 7;
  t := move and 63;
  tr := (t shr 3) and 7;
  tf := t and 7;
  p := (move shr 12) and 7;
  SetLength(move_s, 4);
  move_s[1] := Chr(ff + Ord('a'));
  move_s[2] := Chr(fr + Ord('1'));
  move_s[3] := Chr(tf + Ord('a'));
  move_s[4] := Chr(tr + Ord('1'));
  if p <> 0 then
  begin
    SetLength(move_s, 5);
    move_s[5] := promote_pieces[p];
  end;
  if      move_s = 'e1h1' then
    move_s := 'e1g1'
  else if move_s = 'e1a1' then
    move_s := 'e1c1'
  else if move_s = 'e8h8' then
    move_s := 'e8g8'
  else if move_s = 'e8a8' then
    move_s := 'e8c8'; 
end;

var
  f: THandle;
  entry: entry_t;
  offset: longint;
  file_name: string;
  key: UInt64;
  entries: array[0..MAX_MOVES - 1] of entry_t;
  count: longint = 0;
  i: longint;
  move_s: string;
  total_weight: longint;

begin
  if ParamCount < 2 then
  begin
    WriteLn('Usage: pg_show <book> <hex key>');
    Halt(1);
  end;
  file_name := ParamStr(1);
  key := StrToInt64('$' + ParamStr(2));
  f := FileOpen(file_name, fmOpenRead);
  if f = THandle(-1) then
  begin
    WriteLn('Cannot open file ', file_name);
    Halt(1);
  end;
  offset := find_key(f, key, entry);
  if entry.key <> key then
  begin
    WriteLn('No such key');
    Halt(1);
  end;
  entries[0] := entry;
  count := 1;
  FileSeek(f, 16 * (offset + 1), fsFromBeginning);
  while TRUE do
  begin
    if not entry_from_file(f, entry) then
      Break;
    if entry.key <> key then
      Break;
    if count = MAX_MOVES then
    begin
      WriteLn('Too many moves');
      Halt(1);
    end;
    entries[count] := entry;
    Inc(count);
  end;
  FileClose(f);
  total_weight := 0;
  for i := 0 to Pred(count) do
    Inc(total_weight, entries[i].weight);
  for i := 0 to Pred(count) do
  begin
    move_to_string(move_s, entries[i].move);
    WriteLn(Format('move=%s weight=%5.2f%%', [move_s, 100 * (entries[i].weight / total_weight)]));
  end;
  Halt(0);
end.
